<?php
// otp_send.php
// Generates and stores a new OTP for a given mobile number,
// invalidating any previously unverified OTPs for that number.
require_once 'db_connect.php'; // Include the database connection file

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $mobile_number = $_POST['mobile_number'] ?? '';

    // Validate mobile number input
    if (empty($mobile_number)) {
        echo json_encode(['status' => 'error', 'message' => 'Mobile number is required.']);
        exit();
    }

    // Sanitize mobile number: remove any non-digit characters
    $mobile_number = preg_replace('/[^0-9]/', '', $mobile_number);
    // Basic length validation for mobile number
    if (strlen($mobile_number) < 10 || strlen($mobile_number) > 15) {
        echo json_encode(['status' => 'error', 'message' => 'Invalid mobile number format.']);
        exit();
    }

    // Generate a random 6-digit OTP
    $otp = str_pad(mt_rand(100000, 999999), 6, '0', STR_PAD_LEFT);

    // Set OTP expiry time (e.g., 5 minutes from now)
    $expiry_time = date('Y-m-d H:i:s', strtotime('+5 minutes'));

    // --- IMPORTANT: Real SMS Gateway Integration would go here ---
    // In a production app, you would integrate with an SMS API service
    // (e.g., Twilio, Msg91, Nexmo) to send this `$otp` to the `$mobile_number`.
    // Example (pseudo-code):
    // try {
    //     $smsService = new SmsGatewayAPI('YOUR_API_KEY');
    //     $smsService->sendSMS($mobile_number, "Your Enquiry App OTP is: " . $otp . ". It expires in 5 minutes.");
    // } catch (Exception $e) {
    //     error_log("SMS sending failed: " . $e->getMessage());
    //     echo json_encode(['status' => 'error', 'message' => 'Failed to send OTP. Please try again.']);
    //     exit();
    // }
    // -------------------------------------------------------------

    // --- CRITICAL CHANGE: Delete any existing UNVERIFIED OTPs for this mobile number. ---
    // This ensures that only the NEWLY generated OTP is valid and prevents issues
    // if a user requests multiple OTPs quickly.
    $stmt_delete_old_otps = $conn->prepare("DELETE FROM otp_codes WHERE mobile_number = ? AND is_verified = FALSE AND expires_at > NOW()");
    $stmt_delete_old_otps->bind_param("s", $mobile_number);
    $stmt_delete_old_otps->execute();
    $stmt_delete_old_otps->close();


    // Store the newly generated OTP in the `otp_codes` table
    $stmt = $conn->prepare("INSERT INTO otp_codes (mobile_number, otp_code, expires_at) VALUES (?, ?, ?)");
    $stmt->bind_param("sss", $mobile_number, $otp, $expiry_time); // 'sss' for three strings

    if ($stmt->execute()) {
        // OTP successfully stored (and theoretically sent via SMS gateway)
        echo json_encode([
            'status' => 'success',
            'message' => 'OTP sent to ' . $mobile_number . '. Please check your phone.',
            // FOR DEVELOPMENT ONLY: Including 'test_otp' for easy testing.
            // REMOVE THIS LINE IN PRODUCTION for security!
            'test_otp' => $otp
        ]);
    } else {
        // Error storing OTP
        error_log("Failed to generate OTP: " . $stmt->error); // Log the actual error
        echo json_encode(['status' => 'error', 'message' => 'Failed to generate OTP.']);
    }

    $stmt->close();
} else {
    // Invalid request method
    echo json_encode(['status' => 'error', 'message' => 'Invalid request method.']);
}

$conn->close();
?>