<?php
// otp_verify.php
// Verifies the entered OTP against the stored OTP for a given mobile number.
// This version includes more specific error messages for better debugging.
require_once 'db_connect.php'; // Include the database connection file

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $mobile_number = $_POST['mobile_number'] ?? '';
    $entered_otp = $_POST['otp'] ?? '';

    // Validate inputs: ensure mobile number and OTP are provided.
    if (empty($mobile_number) || empty($entered_otp)) {
        echo json_encode(['status' => 'error', 'message' => 'Mobile number and OTP are required.']);
        exit();
    }

    // Sanitize mobile number: remove any non-digit characters to ensure consistency.
    $mobile_number = preg_replace('/[^0-9]/', '', $mobile_number);

    // --- Diagnostic Step: First, check for any OTPs for this mobile number. ---
    // We fetch the latest OTP regardless of its `is_verified` status or `expires_at` time
    // to provide more specific feedback to the user.
    $stmt_check_any = $conn->prepare("SELECT id, otp_code, expires_at, is_verified FROM otp_codes WHERE mobile_number = ? ORDER BY created_at DESC LIMIT 1");
    $stmt_check_any->bind_param("s", $mobile_number); // 's' for string parameter
    $stmt_check_any->execute(); // Execute the prepared statement
    $result_check_any = $stmt_check_any->get_result(); // Get the result set

    // Check if any OTP record was found for the given mobile number.
    if ($result_check_any->num_rows > 0) {
        $otp_record = $result_check_any->fetch_assoc(); // Fetch the OTP record as an associative array
        $stored_otp = $otp_record['otp_code']; // The OTP code stored in the database
        $otp_id = $otp_record['id']; // The ID of the OTP record
        $expires_at = strtotime($otp_record['expires_at']); // Convert expiry time to Unix timestamp for comparison
        $is_verified = $otp_record['is_verified']; // Boolean status of whether OTP has been verified
        $current_time = time(); // Current Unix timestamp

        // --- Detailed Error Checks ---

        // 1. Check if the OTP has already been used/verified.
        if ($is_verified) {
            echo json_encode(['status' => 'error', 'message' => 'OTP has already been used. Please request a new one.']);
        }
        // 2. Check if the OTP has expired.
        else if ($current_time > $expires_at) {
            echo json_encode(['status' => 'error', 'message' => 'OTP has expired. Please request a new one.']);
        }
        // 3. If not verified and not expired, then check if the entered OTP matches the stored OTP.
        else if ($entered_otp === $stored_otp) {
            // OTP is correct and valid.
            // Mark the OTP as verified in the database to prevent its reuse.
            $stmt_update = $conn->prepare("UPDATE otp_codes SET is_verified = TRUE WHERE id = ?");
            $stmt_update->bind_param("i", $otp_id); // 'i' for integer parameter
            $stmt_update->execute(); // Execute the update statement
            $stmt_update->close(); // Close the update statement

            echo json_encode(['status' => 'success', 'message' => 'OTP verified successfully.']);
        } else {
            // OTP does not match, but it was otherwise active (not expired, not used).
            echo json_encode(['status' => 'error', 'message' => 'Invalid OTP.']);
        }
    } else {
        // No OTP record found for this mobile number at all.
        echo json_encode(['status' => 'error', 'message' => 'No OTP found for this mobile number. Please send OTP first.']);
    }

    $stmt_check_any->close(); // Close the select statement
} else {
    // If the request method is not POST, return an error.
    echo json_encode(['status' => 'error', 'message' => 'Invalid request method.']);
}

$conn->close(); // Close the database connection
?>