<?php
// floors.php - Floors Management Page
session_start();

// Check if the admin is NOT logged in
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: admin_login.php');
    exit;
}
$pageTitle = 'Floors - Hostel Management';
require_once 'config.php'; // Include the database configuration
include_once '_header.php';
include_once '_sidebar.php';
// --- PHP Logic to Fetch Hostels and Floors Data ---
$hostels = []; // For the dropdown
$floors = [];  // For the table
try {
    // Fetch all hostels for the dropdown in the add floor form
    $sqlHostels = "SELECT hostel_id, name FROM hostels ORDER BY name ASC";
    $stmtHostels = $pdo->prepare($sqlHostels);
    $stmtHostels->execute();
    $hostels = $stmtHostels->fetchAll(PDO::FETCH_ASSOC);
    // Fetch all floors with their associated hostel name
    $sqlFloors = "SELECT f.floor_id, f.hostel_id, h.name AS hostel_name, f.floor_number, f.description, f.total_rooms, f.created_at
                  FROM floors f
                  JOIN hostels h ON f.hostel_id = h.hostel_id
                  ORDER BY h.name, f.floor_number ASC";
    $stmtFloors = $pdo->prepare($sqlFloors);
    $stmtFloors->execute();
    $floors = $stmtFloors->fetchAll(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    error_log("Error fetching data for floors.php: " . $e->getMessage());
    $errorMessage = "Could not load data. Please try again later.";
}
?>
    <main class="container-fluid" id="appContent">
        <h1 class="mb-4">Floor Management</h1>

        <!-- Button to Toggle Add New Floor Form -->
        <div class="d-flex justify-content-end mb-3">
            <button class="btn btn-primary" id="toggleAddFloorFormBtn">
                <i class="fas fa-plus me-2"></i> Add New Floor
            </button>
        </div>
        <!-- Add Floor Form -->
        <div class="card mb-4" id="addFloorFormCard" style="display: none;">
            <div class="card-header">
                Add New Floor
            </div>
            <div class="card-body">
                <form id="addFloorForm">
                    <div class="row g-3">
                        <div class="col-md-4">
                            <label for="selectHostelFloor" class="form-label">Select Hostel <span class="text-danger">*</span></label>
                            <select class="form-select" id="selectHostelFloor" name="hostelId" required>
                                <option value="">Choose Hostel...</option>
                                <?php foreach ($hostels as $hostel): ?>
                                    <option value="<?php echo htmlspecialchars($hostel['hostel_id']); ?>">
                                        <?php echo htmlspecialchars($hostel['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <?php if (isset($hostelsDropdownError)): ?>
                                <div class="text-danger mt-1"><?php echo htmlspecialchars($hostelsDropdownError); ?></div>
                            <?php endif; ?>
                        </div>
                        <div class="col-md-4">
                            <label for="floorNumber" class="form-label">Floor Number <span class="text-danger">*</span></label>
                            <input type="number" class="form-control" id="floorNumber" name="floorNumber" min="0" required>
                        </div>
                        <div class="col-md-4">
                            <label for="totalRooms" class="form-label">Total Rooms on Floor</label>
                            <input type="number" class="form-control" id="totalRooms" name="totalRooms" min="0" value="0">
                        </div>
                        <div class="col-12">
                            <label for="floorDescription" class="form-label">Description (Optional)</label>
                            <input type="text" class="form-control" id="floorDescription" name="description" placeholder="e.g., Ground Floor, Girls' Wing">
                        </div>
                        <div class="col-12 text-end">
                            <button type="submit" class="btn btn-primary"><i class="fas fa-plus"></i> Add Floor</button>
                        </div>
                    </div>
                </form>
                <div id="floorMessage" class="mt-3 alert d-none" role="alert">
                    <!-- Message will be displayed here -->
                </div>
            </div>
        </div>

        <div class="card">
            <div class="card-header">
                View/Manage Floors
            </div>
            <div class="card-body">
                <?php if (isset($errorMessage)): ?>
                    <div class="alert alert-danger" role="alert">
                        <?php echo htmlspecialchars($errorMessage); ?>
                    </div>
                <?php else: ?>
                    <?php if (empty($floors)): ?>
                        <div class="alert alert-info" role="alert">
                            No floors found. Add new floors using the form above.
                        </div>
                    <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-striped table-hover">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Hostel Name</th>
                                        <th>Floor Number</th>
                                        <th>Description</th>
                                        <th>Total Rooms</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($floors as $floor): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($floor['floor_id']); ?></td>
                                            <td><?php echo htmlspecialchars($floor['hostel_name']); ?></td>
                                            <td><?php echo htmlspecialchars($floor['floor_number']); ?></td>
                                            <td><?php echo htmlspecialchars($floor['description'] ?: 'N/A'); ?></td>
                                            <td><?php echo htmlspecialchars($floor['total_rooms']); ?></td>
                                            <td>
                                                <button class="btn btn-sm btn-warning rounded-pill me-1 edit-floor-btn" data-id="<?php echo htmlspecialchars($floor['floor_id']); ?>">
                                                    <i class="fas fa-edit"></i> Edit
                                                </button>
                                                <button class="btn btn-sm btn-danger rounded-pill delete-floor-btn" data-id="<?php echo htmlspecialchars($floor['floor_id']); ?>" data-floor-number="<?php echo htmlspecialchars($floor['floor_number']); ?>" data-hostel-name="<?php echo htmlspecialchars($floor['hostel_name']); ?>">
                                                    <i class="fas fa-trash-alt"></i> Delete
                                                </button>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
        </div>
    </main>

    <!-- Edit Floor Modal (Placeholder for future functionality) -->
    <div class="modal fade" id="editFloorModal" tabindex="-1" aria-labelledby="editFloorModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="editFloorModalLabel">Edit Floor Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <form id="editFloorForm">
                        <input type="hidden" id="editFloorId" name="floor_id">
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label for="editHostelSelect" class="form-label">Hostel <span class="text-danger">*</span></label>
                                <select class="form-select" id="editHostelSelect" name="hostelId" required>
                                    <option value="">Choose Hostel...</option>
                                    <?php foreach ($hostels as $hostel): ?>
                                        <option value="<?php echo htmlspecialchars($hostel['hostel_id']); ?>">
                                            <?php echo htmlspecialchars($hostel['name']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-6">
                                <label for="editFloorNumber" class="form-label">Floor Number <span class="text-danger">*</span></label>
                                <input type="number" class="form-control" id="editFloorNumber" name="floorNumber" min="0" required>
                            </div>
                            <div class="col-12">
                                <label for="editFloorDescription" class="form-label">Description (Optional)</label>
                                <textarea class="form-control" id="editFloorDescription" name="description" rows="2"></textarea>
                            </div>
                            <div class="col-12">
                                <label for="editTotalRooms" class="form-label">Total Rooms on Floor</label>
                                <input type="number" class="form-control" id="editTotalRooms" name="totalRooms" min="0">
                            </div>
                        </div>
                    </form>
                    <div id="editFloorMessage" class="mt-3 alert d-none" role="alert">
                        <!-- Message will be displayed here -->
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="button" class="btn btn-primary" id="saveEditFloorBtn">Save changes</button>
                </div>
            </div>
        </div>
    </div>


    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const toggleAddFloorFormBtn = document.getElementById('toggleAddFloorFormBtn');
            const addFloorFormCard = document.getElementById('addFloorFormCard');
            const addFloorForm = document.getElementById('addFloorForm');
            const floorMessage = document.getElementById('floorMessage');

            // Edit Modal elements
            const editFloorModal = new bootstrap.Modal(document.getElementById('editFloorModal'));
            const editFloorForm = document.getElementById('editFloorForm');
            const editFloorIdInput = document.getElementById('editFloorId');
            const editHostelSelect = document.getElementById('editHostelSelect');
            const editFloorNumberInput = document.getElementById('editFloorNumber');
            const editFloorDescriptionInput = document.getElementById('editFloorDescription');
            const editTotalRoomsInput = document.getElementById('editTotalRooms');
            const saveEditFloorBtn = document.getElementById('saveEditFloorBtn');
            const editFloorMessage = document.getElementById('editFloorMessage');


            // Function to display messages (for Add Floor Form)
            function showMessage(message, type) {
                floorMessage.textContent = message;
                floorMessage.className = `mt-3 alert alert-${type}`; // alert-success, alert-danger, etc.
                floorMessage.classList.remove('d-none');
                setTimeout(() => {
                    floorMessage.classList.add('d-none');
                }, 5000); // Hide message after 5 seconds
            }

            // Function to display messages (for Edit Floor Modal)
            function showEditMessage(message, type) {
                editFloorMessage.textContent = message;
                editFloorMessage.className = `mt-3 alert alert-${type}`;
                editFloorMessage.classList.remove('d-none');
                setTimeout(() => {
                    editFloorMessage.classList.add('d-none');
                }, 5000);
            }


            // Toggle Add Floor Form visibility
            toggleAddFloorFormBtn.addEventListener('click', function() {
                if (addFloorFormCard.style.display === 'none') {
                    addFloorFormCard.style.display = 'block';
                    toggleAddFloorFormBtn.innerHTML = '<i class="fas fa-minus me-2"></i> Hide Form';
                } else {
                    addFloorFormCard.style.display = 'none';
                    toggleAddFloorFormBtn.innerHTML = '<i class="fas fa-plus me-2"></i> Add New Floor';
                    addFloorForm.reset(); // Clear form when hidden
                    addFloorForm.classList.remove('was-validated'); // Remove validation styling
                    floorMessage.classList.add('d-none'); // Hide any previous messages
                }
            });

            // Handle Add Floor Form submission
            addFloorForm.addEventListener('submit', async function(event) {
                event.preventDefault(); // Prevent actual form submission

                // Basic client-side validation (Bootstrap's built-in validation)
                if (!addFloorForm.checkValidity()) {
                    event.stopPropagation();
                    addFloorForm.classList.add('was-validated'); // Add Bootstrap validation styling
                    showMessage('Please fill all required fields correctly.', 'warning');
                    return;
                } else {
                    addFloorForm.classList.remove('was-validated'); // Remove validation styling if valid
                }

                // Collect form data
                const formData = new FormData(addFloorForm);
                const floorData = {};
                for (let [key, value] of formData.entries()) {
                    floorData[key] = value;
                }
                // Ensure floorNumber and totalRooms are integers
                floorData['floorNumber'] = parseInt(floorData['floorNumber']);
                floorData['totalRooms'] = parseInt(floorData['totalRooms']);


                console.log('Submitting Add Floor data:', floorData);
                showMessage('Adding floor...', 'info'); // Show loading message

                try {
                    const response = await fetch('add_floor_process.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json', // Tell the server we're sending JSON
                        },
                        body: JSON.stringify(floorData), // Convert JS object to JSON string
                    });

                    const result = await response.json(); // Parse the JSON response from the server

                    if (result.success) {
                        showMessage(result.message, 'success');
                        addFloorForm.reset(); // Clear the form on success
                        addFloorForm.classList.remove('was-validated'); // Clear validation styling
                        // IMPORTANT: Reload the page to show the newly added floor in the table
                        setTimeout(() => {
                            location.reload();
                        }, 1000); // Reload after 1 second to give message time to show
                    } else {
                        showMessage(result.message, 'danger');
                    }
                } catch (error) {
                    console.error('Error adding floor:', error);
                    showMessage('An unexpected error occurred. Please try again.', 'danger');
                }
            });

            // --- JavaScript for Action Buttons (Edit/Delete) ---
            // Event delegation for edit and delete buttons
            document.querySelector('table').addEventListener('click', async function(event) {
                // Check if the clicked element or its parent is an edit button
                if (event.target.closest('.edit-floor-btn')) {
                    const button = event.target.closest('.edit-floor-btn');
                    const floorId = button.dataset.id;
                    console.log('Edit button clicked for Floor ID:', floorId);

                    // Fetch floor data for editing
                    try {
                        const response = await fetch(`get_floor_details.php?id=${floorId}`); // This will be a new PHP file
                        const data = await response.json();

                        if (data.success && data.floor) {
                            const floor = data.floor;
                            editFloorIdInput.value = floor.floor_id;
                            editHostelSelect.value = floor.hostel_id; // Set selected hostel
                            editFloorNumberInput.value = floor.floor_number;
                            editFloorDescriptionInput.value = floor.description;
                            editTotalRoomsInput.value = floor.total_rooms; // Set total rooms

                            editFloorModal.show(); // Show the modal
                            showEditMessage('', ''); // Clear any previous messages in the modal
                            editFloorForm.classList.remove('was-validated'); // Clear validation styling
                        } else {
                            showEditMessage(data.message || 'Failed to fetch floor details.', 'danger');
                        }
                    } catch (error) {
                        console.error('Error fetching floor details:', error);
                        showEditMessage('An error occurred while fetching details.', 'danger');
                    }
                }

                // Check if the clicked element or its parent is a delete button
                if (event.target.closest('.delete-floor-btn')) {
                    const button = event.target.closest('.delete-floor-btn');
                    const floorId = button.dataset.id;
                    const floorNumber = button.dataset.floorNumber;
                    const hostelName = button.dataset.hostelName;

                    // Use a custom alert/confirm for better UI, as window.confirm is blocking
                    const confirmDelete = confirm(`Are you sure you want to delete Floor No. ${floorNumber} from ${hostelName} (ID: ${floorId})? This action cannot be undone.`);

                    if (confirmDelete) {
                        console.log('Confirmed deletion for Floor ID:', floorId);
                        showMessage('Deleting floor...', 'info'); // Show loading message on main page

                        try {
                            // THIS IS THE LINE THAT WAS PREVIOUSLY COMMENTED/REPLACED BY ALERT
                            const response = await fetch('delete_floor_process.php', {
                                method: 'POST',
                                headers: {
                                    'Content-Type': 'application/json',
                                },
                                body: JSON.stringify({ floor_id: floorId }),
                            });

                            const result = await response.json();

                            if (result.success) {
                                showMessage(result.message, 'success');
                                // Reload the page to show updated data
                                setTimeout(() => {
                                    location.reload();
                                }, 1000);
                            } else {
                                showMessage(result.message, 'danger');
                            }
                        } catch (error) {
                            console.error('Error deleting floor:', error);
                            showMessage('An unexpected error occurred during deletion. Please try again.', 'danger');
                        }
                    } else {
                        showMessage('Deletion cancelled.', 'secondary');
                    }
                }
            });

            // Handle Save Changes button in Edit Modal
            saveEditFloorBtn.addEventListener('click', async function() {
                // Basic client-side validation for edit form
                if (!editFloorForm.checkValidity()) {
                    event.stopPropagation();
                    editFloorForm.classList.add('was-validated');
                    showEditMessage('Please fill all required fields correctly.', 'warning');
                    return;
                } else {
                    editFloorForm.classList.remove('was-validated');
                }

                const formData = new FormData(editFloorForm);
                const updatedFloorData = {};
                for (let [key, value] of formData.entries()) {
                    updatedFloorData[key] = value;
                }
                updatedFloorData['floorNumber'] = parseInt(updatedFloorData['floorNumber']); // Ensure integer type
                updatedFloorData['totalRooms'] = parseInt(updatedFloorData['totalRooms']); // Ensure integer type

                console.log('Submitting updated floor data:', updatedFloorData);
                showEditMessage('Saving changes...', 'info');

                try {
                    const response = await fetch('update_floor_process.php', { // This will be a new PHP file
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        },
                        body: JSON.stringify(updatedFloorData),
                    });

                    const result = await response.json();

                    if (result.success) {
                        showEditMessage(result.message, 'success');
                        // Reload the page to show updated data in the table
                        setTimeout(() => {
                            editFloorModal.hide(); // Hide modal first
                            location.reload();
                        }, 1000);
                    } else {
                        showEditMessage(result.message, 'danger');
                    }
                } catch (error) {
                    console.error('Error updating floor:', error);
                    showEditMessage('An unexpected error occurred while saving changes. Please try again.', 'danger');
                }
            });
        });
    </script>
<?php include_once 'footer.php'; ?>
