<?php
// Start the session
session_start();
// Check if the user is logged in, if not then redirect to login page
if (!isset($_SESSION["loggedin"]) || $_SESSION["loggedin"] !== true) {
    header("location: ../login.php");
    exit;
}
// Check if the user is an admin, if not then redirect to employee dashboard
if ($_SESSION["role"] !== "admin") {
    header("location: ../employee/dashboard.php");
    exit;
}
// Include config file
require_once "../config/db.php";

// Define variables and initialize with empty values
$username = $email = $password = $confirm_password = $contact_number = $department_id = "";
$username_err = $email_err = $password_err = $confirm_password_err = $contact_number_err = $department_id_err = "";

// Fetch departments from the database
$departments = [];
$sql_departments = "SELECT department_id, department_name FROM departments ORDER BY department_name ASC";
if ($result_departments = mysqli_query($link, $sql_departments)) {
    if (mysqli_num_rows($result_departments) > 0) {
        while ($row_department = mysqli_fetch_assoc($result_departments)) {
            $departments[] = $row_department;
        }
        mysqli_free_result($result_departments);
    }
} else {
    echo "Oops! Something went wrong. Please try again later.";
}
// Processing form data when form is submitted
if ($_SERVER["REQUEST_METHOD"] == "POST") {

    // Validate username
    if (empty(trim($_POST["username"]))) {
        $username_err = "Please enter a username.";
    } else {
        // Prepare a select statement
        $sql = "SELECT user_id FROM users WHERE username = ?";

        if ($stmt = mysqli_prepare($link, $sql)) {
            // Bind variables to the prepared statement as parameters
            mysqli_stmt_bind_param($stmt, "s", $param_username);

            // Set parameters
            $param_username = trim($_POST["username"]);

            // Attempt to execute the prepared statement
            if (mysqli_stmt_execute($stmt)) {
                // Store result
                mysqli_stmt_store_result($stmt);

                if (mysqli_stmt_num_rows($stmt) == 1) {
                    $username_err = "This username is already taken.";
                } else {
                    $username = trim($_POST["username"]);
                }
            } else {
                echo "Oops! Something went wrong. Please try again later.";
            }

            // Close statement
            mysqli_stmt_close($stmt);
        }
    }

    // Validate email
    if (empty(trim($_POST["email"]))) {
        $email_err = "Please enter an email.";
    } else {
        // Prepare a select statement
        $sql = "SELECT user_id FROM users WHERE email = ?";

        if ($stmt = mysqli_prepare($link, $sql)) {
            // Bind variables to the prepared statement as parameters
            mysqli_stmt_bind_param($stmt, "s", $param_email);

            // Set parameters
            $param_email = trim($_POST["email"]);

            // Attempt to execute the prepared statement
            if (mysqli_stmt_execute($stmt)) {
                // Store result
                mysqli_stmt_store_result($stmt);

                if (mysqli_stmt_num_rows($stmt) == 1) {
                    $email_err = "This email is already registered.";
                } else {
                    $email = trim($_POST["email"]);
                }
            } else {
                echo "Oops! Something went wrong. Please try again later.";
            }

            // Close statement
            mysqli_stmt_close($stmt);
        }
    }

    // Validate password
    if (empty(trim($_POST["password"]))) {
        $password_err = "Please enter a password.";
    } elseif (strlen(trim($_POST["password"])) < 6) {
        $password_err = "Password must have at least 6 characters.";
    } else {
        $password = trim($_POST["password"]);
    }

    // Validate confirm password
    if (empty(trim($_POST["confirm_password"]))) {
        $confirm_password_err = "Please confirm password.";
    } else {
        $confirm_password = trim($_POST["confirm_password"]);
        if (empty($password_err) && ($password != $confirm_password)) {
            $confirm_password_err = "Password did not match.";
        }
    }

    // Validate contact number
    if (empty(trim($_POST["contact_number"]))) {
        $contact_number_err = "Please enter a contact number.";
    } else {
        $contact_number = trim($_POST["contact_number"]);
    }
    // Validate department
    if (empty(trim($_POST["department_id"]))) {
        $department_id_err = "Please select a department.";
    } else {
        $department_id = trim($_POST["department_id"]);
    }
    // Check input errors before inserting in database
    if (empty($username_err) && empty($email_err) && empty($password_err) && empty($confirm_password_err) && empty($contact_number_err) && empty($department_id_err)) {
        // Prepare an insert statement
        $sql = "INSERT INTO users (username, email, password, contact_number, role, department_id) VALUES (?, ?, ?, ?, 'employee', ?)";
        if ($stmt = mysqli_prepare($link, $sql)) {
            // Bind variables to the prepared statement as parameters
            mysqli_stmt_bind_param($stmt, "ssssi", $param_username, $param_email, $param_password, $param_contact_number, $param_department_id);
            // Set parameters
            $param_username = $username;
            $param_email = $email;
            $param_password = password_hash($password, PASSWORD_DEFAULT); // Creates a password hash
            $param_contact_number = $contact_number;
            $param_department_id = $department_id;
            // Attempt to execute the prepared statement
            if (mysqli_stmt_execute($stmt)) {
                // Redirect to employee management page
                header("location: dashboard.php");
            } else {
                echo "Something went wrong. Please try again later.";
            }
            // Close statement
            mysqli_stmt_close($stmt);
        }
    }
    // Close connection
    mysqli_close($link);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Add Employee</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css">
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: #f4f7f6;
            color: #333;
        }
        .wrapper {
            display: flex;
            min-height: 100vh;
            transition: all 0.3s ease;
        }
        .sidebar {
            width: 250px;
            background-color: #2c3e50;
            color: #ecf0f1;
            padding: 20px;
            box-shadow: 2px 0 5px rgba(0,0,0,0.1);
            display: flex;
            flex-direction: column;
            transition: all 0.3s ease;
        }
        .sidebar h3 {
            text-align: center;
            margin-bottom: 30px;
            color: #ecf0f1;
        }
        .sidebar ul {
            list-style: none;
            padding: 0;
            flex-grow: 1;
        }
        .sidebar ul li {
            margin-bottom: 15px;
        }
        .sidebar ul li a {
            color: #ecf0f1;
            text-decoration: none;
            padding: 10px 15px;
            display: block;
            border-radius: 5px;
            transition: background-color 0.3s ease;
        }
        .sidebar ul li a:hover,
        .sidebar ul li a.active {
            background-color: #34495e;
        }
        .main-content {
            flex-grow: 1;
            padding: 30px;
            transition: all 0.3s ease;
        }
        .navbar {
            background-color: #ffffff;
            padding: 15px 30px;
            border-bottom: 1px solid #ddd;
            display: flex;
            justify-content: space-between;
            align-items: center;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
        }
        .navbar h2 {
            margin: 0;
            color: #333;
        }
        .navbar .profile-info {
            display: flex;
            align-items: center;
        }
        .navbar .profile-info .username {
            margin-right: 15px;
            font-weight: bold;
            color: #555;
        }
        .navbar .btn-logout {
            background-color: #e74c3c;
            color: #fff;
            border: none;
            padding: 8px 15px;
            border-radius: 5px;
            text-decoration: none;
            transition: background-color 0.3s ease;
        }
        .navbar .btn-logout:hover {
            background-color: #c0392b;
        }
        .card {
            background-color: #fff;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.08);
            margin-bottom: 30px;
            padding: 20px;
        }
        .card-header {
            font-size: 1.5rem;
            margin-bottom: 20px;
            color: #2c3e50;
            border-bottom: 1px solid #eee;
            padding-bottom: 10px;
        }
        .form-group {
            margin-bottom: 15px;
        }
        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }
        .form-control {
            width: 100%;
            padding: 10px;
            border: 1px solid #ced4da;
            border-radius: 4px;
            box-sizing: border-box;
        }
        .btn-primary {
            background-color: #007bff;
            border-color: #007bff;
            color: #fff;
            padding: 10px 15px;
            border-radius: 4px;
            cursor: pointer;
        }
        .btn-primary:hover {
            background-color: #0056b3;
            border-color: #0056b3;
        }
        .btn-secondary {
            background-color: #6c757d;
            border-color: #6c757d;
            color: #fff;
            padding: 10px 15px;
            border-radius: 4px;
            cursor: pointer;
            margin-left: 10px;
        }
        .btn-secondary:hover {
            background-color: #5a6268;
            border-color: #545b62;
        }
        .help-block {
            color: #dc3545;
            font-size: 0.875em;
            margin-top: 5px;
        }
        /* Responsive styles */
        @media (max-width: 768px) {
            .wrapper {
                flex-direction: column;
            }
            .sidebar {
                height: auto;
                padding: 15px;
                box-shadow: none;
                         box-shadow: none;
                    position: fixed;
                    top: 0;
                    left: 0;
                    width: 100%;
                    z-index: 999; 
            }
            .sidebar ul {
                display: none; /* Hidden by default */
                flex-direction: column;
            }
            .sidebar.active ul {
                display: flex; /* Show when active */
            }
            .sidebar .sidebar-header {
                display: flex;
                justify-content: space-between;
                align-items: center;
            }
            .sidebar .sidebar-header h3 {
                margin-bottom: 0;
            }
            .main-content {
                padding: 15px;
            }
            .navbar {
                flex-direction: column;
                align-items: flex-start;
                padding: 15px;
            }
            .navbar h2 {
                margin-bottom: 10px;
            }
            .navbar .profile-info {
                margin-top: 10px;
                width: 100%;
                justify-content: space-between;
            }
            .navbar .profile-info .username {
                margin-right: 0;
            }
        }
    </style>
</head>
<body>
    <div class="wrapper" id="wrapper">
        <nav class="sidebar" id="sidebar">
            <div class="sidebar-header">
                <h3>Work Allocation</h3>
                <button type="button" id="sidebarCollapse" class="btn btn-dark d-md-none">
                    <i class="bi bi-list"></i>
                </button>
            </div>
            <ul>
                <li><a href="dashboard.php"><i class="bi bi-grid-fill"></i> Dashboard</a></li>
                <li><a href="manage_employees.php" class="active"><i class="bi bi-people-fill"></i> Manage Employees</a></li>
                <li><a href="manage_tasks.php"><i class="bi bi-list-task"></i> Manage Tasks</a></li>
                <li><a href="../logout.php"><i class="bi bi-box-arrow-right"></i> Logout</a></li>
            </ul>
        </nav>
        <div class="main-content">
            <br>
            <header class="navbar mt-5">
                <h2>Add New Employee</h2>
                <div class="profile-info">
                    <span class="username">Welcome, <?php echo htmlspecialchars($_SESSION["username"]); ?></span>
                    <a href="../logout.php" class="btn-logout">Logout</a>
                </div>
            </header>
            <div class="container-fluid mt-4" style="padding-top: 70px;"> <!-- Add padding to prevent content from being hidden by fixed navbar -->
                <div class="card">
                    <div class="card-header">Add New Employee</div>
                    <div class="card-body">
                        <form action="<?php echo htmlspecialchars($_SERVER["PHP_SELF"]); ?>" method="post">
                            <div class="form-group">
                                <label>Username</label>
                                <input type="text" name="username" class="form-control <?php echo (!empty($username_err)) ? 'is-invalid' : ''; ?>" value="<?php echo $username; ?>">
                                <span class="help-block"><?php echo $username_err; ?></span>
                            </div>
                            <div class="form-group">
                                <label>Email</label>
                                <input type="email" name="email" class="form-control <?php echo (!empty($email_err)) ? 'is-invalid' : ''; ?>" value="<?php echo $email; ?>">
                                <span class="help-block"><?php echo $email_err; ?></span>
                            </div>
                            <div class="form-group">
                                <label>Password</label>
                                <input type="password" name="password" class="form-control <?php echo (!empty($password_err)) ? 'is-invalid' : ''; ?>" value="<?php echo $password; ?>">
                                <span class="help-block"><?php echo $password_err; ?></span>
                            </div>
                            <div class="form-group">
                                <label>Confirm Password</label>
                                <input type="password" name="confirm_password" class="form-control <?php echo (!empty($confirm_password_err)) ? 'is-invalid' : ''; ?>">
                                <span class="help-block"><?php echo $confirm_password_err; ?></span>
                            </div>
                            <div class="form-group">
                                <label>Contact Number</label>
                                <input type="text" name="contact_number" class="form-control <?php echo (!empty($contact_number_err)) ? 'is-invalid' : ''; ?>" value="<?php echo $contact_number; ?>">
                                <span class="help-block"><?php echo $contact_number_err; ?></span>
                            </div>
                            <div class="form-group">
                                <label>Department</label>
                                <select name="department_id" class="form-control <?php echo (!empty($department_id_err)) ? 'is-invalid' : ''; ?>">
                                    <option value="">Select Department</option>
                                    <?php
                                    foreach ($departments as $department) {
                                        echo '<option value="' . $department['department_id'] . '"' . (($department_id == $department['department_id']) ? 'selected' : '') . '>' . htmlspecialchars($department['department_name']) . '</option>';
                                    }
                                    ?>
                                </select>
                                <span class="help-block"><?php echo $department_id_err; ?></span>
                            </div>
                            <div class="form-group">
                                <input type="submit" class="btn btn-primary" value="Submit">
                                <a href="dashboard.php" class="btn btn-secondary">Cancel</a>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Sidebar toggle for mobile
        document.getElementById('sidebarCollapse').addEventListener('click', function() {
            document.getElementById('sidebar').classList.toggle('active');
        });
    </script>
</body>
</html>